package com.pcloud.common.utils.exceptioin;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.function.Failable;
import org.apache.commons.lang3.function.FailableRunnable;
import org.apache.commons.lang3.function.FailableSupplier;

import java.util.function.Consumer;

/**
 * 异常处理工具类
 */
@Slf4j
public final class ExceptionUtils {

    private ExceptionUtils() {
    }

    /**
     * 出现错误时，返回默认值
     *
     * @param supplier     （可能抛出异常的）带返回值的操作
     * @param onError      错误时的处理方法
     * @param defaultValue 错误时返回的默认值
     * @param <T>          类型
     * @return 成功返回执行结果，失败返回默认值
     */
    public static <T> T defaultWhenException(final FailableSupplier<T, ? extends Exception> supplier, final Consumer<Exception> onError, final T defaultValue) {
        try {
            return Failable.get(supplier);
        } catch (final Exception e) {
            onError.accept(e);
            return defaultValue;
        }
    }


    /**
     * 出现错误时，返回默认值, 默认打印warn日志
     *
     * @param supplier     （可能抛出异常的）带返回值的操作
     * @param defaultValue 错误时返回的默认值
     * @param <T>          类型
     * @return 成功返回执行结果，失败返回默认值
     */
    public static <T> T defaultWhenException(final FailableSupplier<T, ? extends Exception> supplier, final T defaultValue) {
        try {
            return Failable.get(supplier);
        } catch (final Exception e) {
            log.warn("[{}] 操作失败！ ERR：{}", Thread.currentThread().getStackTrace()[2].getMethodName(), e.getMessage(), e);
            return defaultValue;
        }
    }

    /**
     * 出现错误时，执行指定方法
     *
     * @param runnable （可能抛出异常的）无返回值的操作
     * @param onError  错误时无返回值的处理方法
     */
    public static void runWhenException(final FailableRunnable<? extends Exception> runnable, final Consumer<Exception> onError) {
        try {
            Failable.run(runnable);
        } catch (final Exception e) {
            onError.accept(e);
        }
    }

    /**
     * 出现错误时,忽略异常，打印warn日志
     *
     * @param runnable （可能抛出异常的）无返回值的操作
     */
    public static void runWhenException(final FailableRunnable<? extends Exception> runnable) {
        try {
            Failable.run(runnable);
        } catch (final Exception e) {
            log.warn("[{}] 操作失败！ ERR：{}", Thread.currentThread().getStackTrace()[2].getMethodName(), e.getMessage(), e);
        }
    }
}
