/**
 *
 */
package com.pcloud.book.book.facade.impl;

import com.google.common.collect.Lists;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.book.biz.BookBiz;
import com.pcloud.book.book.dto.AdviserBookInfoDTO;
import com.pcloud.book.book.dto.BookAssocCount;
import com.pcloud.book.book.dto.BookDto;
import com.pcloud.book.book.entity.Book;
import com.pcloud.book.book.facade.BookFacade;
import com.pcloud.book.book.vo.BookQualifyVO;
import com.pcloud.book.book.vo.ListNoAuthBookParam;
import com.pcloud.book.book.vo.ListNoAuthGroupBookParam;
import com.pcloud.book.book.vo.SearchBookVO;
import com.pcloud.book.book.vo.request.BookQualifyInfoVO;
import com.pcloud.book.book.vo.request.BookSearchParamVO;
import com.pcloud.book.book.vo.request.UpdateBookRobotProcessDto;
import com.pcloud.common.core.constant.SystemCode;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.permission.Permission;
import com.pcloud.common.permission.PermissionCode.ServerPermissionCode;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;

import com.pcloud.wechatgroup.message.enums.IsSystem;

import io.swagger.annotations.ApiOperation;

import org.codehaus.jackson.JsonParseException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @描述：书籍接口实现类
 * @作者：songx
 * @创建时间：2016年12月23日,下午3:28:46 @版本：1.0
 */
@RequestMapping("/book")
@RestController("bookFacade")
public class BookFacadeImpl implements BookFacade {

    @Autowired
    private BookBiz bookBiz;

    /**
     * 创建书籍
     */
    @Override
    @RequestMapping(value = "create", method = RequestMethod.POST)
    public ResponseDto<BookDto> create( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.TENANT_ID);
        String systemCode = SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE).toString();
        // 实体赋值
        book.setCreatedUser(adviserId);
        book.setAgentId(agentId);
        BookDto bookDto = bookBiz.create(book, systemCode);
        return new ResponseDto<>(bookDto);
    }

    /**
     * 修改书籍
     */
    @Override
    @RequestMapping(value = "update", method = RequestMethod.POST)
    public ResponseDto<?> update( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        // 实体赋值
        book.setLastModifiedUser(adviserId);
        bookBiz.update(book);
        return new ResponseDto<>();
    }

    /**
     * 修改书籍封面图
     */
    @Override
    @RequestMapping(value = "updateCoverImg", method = RequestMethod.POST)
    public ResponseDto<?> updateCoverImg( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        // 实体赋值
        book.setLastModifiedUser(adviserId);
        bookBiz.updateCoverImg(book);
        return new ResponseDto<>();
    }

    /**
     * 修改书籍(编辑)
     */
    @Override
    @RequestMapping(value = "updateByAdviser", method = RequestMethod.POST)
    public ResponseDto<BookDto> updateByAdviser( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        // 实体赋值
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.TENANT_ID);
        book.setLastModifiedUser(adviserId);
        book.setCreatedUser(adviserId);
        book.setAgentId(agentId);
        BookDto bookDto = bookBiz.updateByAdviser(book);
        return new ResponseDto<>(bookDto);
    }

    /**
     * 删除书籍
     */
    @Override
    @RequestMapping(value = "delete", method = RequestMethod.GET)
    public ResponseDto<?> delete( @RequestHeader("token") String token,
                                  @RequestParam(value = "bookId") Long bookId )
            throws BizException, PermissionException {
        // token
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookBiz.delete(bookId, partyId);
        return new ResponseDto<>();
    }

    /**
     * 批量删除书籍
     */
    @Override
    @RequestMapping(value = "deletes", method = RequestMethod.GET)
    public ResponseDto<?> deletes( @RequestHeader("token") String token,
                                   @RequestParam(value = "bookIds") String bookIds )
            throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookBiz.deletes(bookIds, partyId);
        return new ResponseDto<>();
    }

    /**
     * 获取单个书籍基础信息(包含统计信息)
     */
    @Override
    @RequestMapping(value = "getById", method = RequestMethod.GET)
    public ResponseDto<BookDto> getById( @RequestHeader("token") String token,
                                         @RequestParam(value = "bookId") Long bookId,
                                         @RequestParam(value = "channelId") Long channelId )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookBiz.getById(bookId, adviserId, channelId));
    }

    /**
     * 获取单个书籍基础信息
     */
    @Override
    @RequestMapping(value = "getBaseById", method = RequestMethod.GET)
    public ResponseDto<BookDto> getBaseById( @RequestParam(value = "bookId") Long bookId )
            throws BizException {
        return new ResponseDto<>(bookBiz.getBaseById(bookId));
    }

    /**
     * 判断书籍ISBN码是否存在
     */
    @Override
    @RequestMapping(value = "isbnExists", method = RequestMethod.GET)
    public ResponseDto<Boolean> isbnExists( @RequestHeader("token") String token,
                                            @RequestParam(value = "typeCode", required = false) String typeCode,
                                            @RequestParam(value = "isbn") String isbn,
                                            @RequestParam(value = "serialNumber", required = false) String serialNumber )
            throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookBiz.isbnExists(typeCode, isbn, serialNumber));
    }

    /**
     * 判断书籍ISBN码是否存在,存在则获取书籍信息
     */
    @Override
    @RequestMapping(value = "isbnExistsOrGetInfo", method = RequestMethod.POST)
    public ResponseDto<BookDto> isbnExistsOrGetInfo( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        book.setCreatedUser(adviserId);
        BookDto journalDto = bookBiz.isbnExistsOrGetInfo(book);
        return new ResponseDto<>(journalDto == null ? new BookDto() : journalDto);
    }

    /**
     * 判断书籍ISBN码是否存在,存在则获取书籍信息
     */
    @Override
    @RequestMapping(value = "isbnExistsOrGetInfo4Wechat", method = RequestMethod.POST)
    public ResponseDto<BookDto> isbnExistsOrGetInfo4Wechat( @CookieValue("userInfo") String userInfo,
                                                            @RequestBody Book book ) throws BizException {
        Long adviserId = Cookie.getId(userInfo, Cookie._PARTY_ID);
        book.setCreatedUser(adviserId);
        BookDto bookDto = bookBiz.isbnExistsOrGetInfo(book);
        if (bookDto == null) {
            bookDto = new BookDto();
            bookDto.setIsbn(book.getIsbn());
            bookDto.setTypeCode(book.getTypeCode());
        }
        return new ResponseDto<>(bookDto);
    }

    /**
     * 获取书籍列表(平台端)
     */
    @Override
    @Permission(Code = ServerPermissionCode.book + "book/listPage")
    @RequestMapping(value = "listPage", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<BookDto>> getListPage( @RequestHeader("token") String token,
                                                          @RequestParam(value = "publish", required = false) String publish,
                                                          @RequestParam(value = "fundName", required = false) String fundName,
                                                          @RequestParam(value = "bookName", required = false) String bookName,
                                                          @RequestParam(value = "isbn", required = false) String isbn,
                                                          @RequestParam(value = "serialNumber", required = false) String serialNumber,
                                                          @RequestParam(value = "isCurrentMonth", required = false) Integer isCurrentMonth,
                                                          @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                          @RequestParam(value = "numPerPage", required = false) Integer numPerPage )
            throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        return new ResponseDto<>(bookBiz.getListPage(isbn, bookName, publish, fundName,
                serialNumber, isCurrentMonth, currentPage, numPerPage));
    }

    /**
     * 获取书籍列表(编辑)
     */
    @Override
    @RequestMapping(value = "listPage4Adviser", method = RequestMethod.GET)
    public ResponseDto<PageBean> getListPage4Adviser( @RequestHeader("token") String token,
                                                      @RequestParam(value = "name", required = false) String name,
                                                      @RequestParam(value = "bookName", required = false) String bookName,
                                                      @RequestParam(value = "isbn", required = false) String isbn,
                                                      @RequestParam(value = "channelId", required = false) Long channelId,
                                                      @RequestParam(value = "typeCode", required = false) String typeCode,
                                                      @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                      @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
                                                      @RequestParam(value = "isMainEditor", required = false) Integer isMainEditor,
                                                      @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
                                                      @RequestParam(value = "bookId", required = false) Integer bookId,
                                                      @RequestParam(value = "templetId", required = false) Long templetId,
                                                      @RequestParam(value = "bookStatus", required = false) Integer bookStatus,
                                                      @RequestParam(value = "secondTempletId", required = false) Long secondTempletId,
                                                      @RequestParam(value = "hasQrcode", required = false) Integer hasQrcode,
                                                      @RequestParam(value = "isPrint", required = false) Boolean isPrint )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("isMainEditor", isMainEditor);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("channelId", channelId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("isbn", isbn != null && "".equals(isbn.trim()) ? null : isbn);
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("bookId", bookId);
        paramMap.put("templetId", templetId);
        paramMap.put("bookStatus", bookStatus);
        paramMap.put("secondTempletId", secondTempletId);
        paramMap.put("secondTempletIds", Optional.ofNullable(paramMap.get("secondTempletId")).map(Lists::newArrayList).orElse(Lists.newArrayList()));
        paramMap.put("isPrint", isPrint);
        PageBean pageBean = bookBiz.getListPage4Adviser(paramMap, new PageParam(currentPage, numPerPage), adviserId, hasQrcode);
        return new ResponseDto<>(pageBean);
    }

    @GetMapping("getList4Adviser")
    @Override
    public ResponseDto<List<BookDto>> getList4Adviser( @RequestHeader("token") String token, @RequestParam(value = "num") Integer num,
                                                       @RequestParam(value = "isPrint", required = false) Boolean isPrint ) throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        List<BookDto> bookDtos = bookBiz.getList4Adviser(adviserId, num, isPrint);
        return new ResponseDto<>(ListUtils.isEmpty(bookDtos) ? Lists.newArrayList() : bookDtos);
    }

    /**
     * 获取书籍详细信息
     */
    @Override
    @RequestMapping(value = "getDetailById", method = RequestMethod.GET)
    public ResponseDto<BookDto> getDetailById( @CookieValue("userInfo") String userInfo,
                                               @RequestParam(value = "bookId", required = false) Long bookId,
                                               @RequestParam(value = "adviserId", required = false) Long adviserId )
            throws BizException {
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        BookDto bookDto = bookBiz.getDetailById(bookId, adviserId, channelId);
        return new ResponseDto<>(bookDto == null ? new BookDto() : bookDto);
    }

    /**
     * 图书收益，获取书籍基础信息(包含统计信息)
     */
    @Override
    @RequestMapping(value = "getById4Profit", method = RequestMethod.GET)
    public ResponseDto<BookDto> getById4Profit( @RequestHeader("token") String token,
                                                @RequestParam(value = "bookId") Long bookId,
                                                @RequestParam(value = "channelId") Long channelId )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        BookDto bookDto = bookBiz.getById4Profit(bookId, adviserId, channelId);
        return new ResponseDto<>(bookDto == null ? new BookDto() : bookDto);
    }

    /**
     * 创建编辑书籍关联关系
     */
    @Override
    @RequestMapping(value = "setBookAdviserRelation", method = RequestMethod.POST)
    public ResponseDto<?> setBookAdviserRelation( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        book.setCreatedUser(adviserId);
        bookBiz.setBookAdviserRelation(book);
        return new ResponseDto<>();
    }

    /**
     * 获取书籍基本信息
     */
    @Override
    @RequestMapping(value = "getBaseInfoById4Wechat", method = RequestMethod.GET)
    public ResponseDto<BookDto> getBaseInfoById4Wechat( @CookieValue("userInfo") String userInfo,
                                                        @RequestParam(value = "bookId") Long bookId )
            throws BizException {
        Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        BookDto bookDto = bookBiz.getBaseInfoById4Wechat(bookId);
        return new ResponseDto<>(bookDto == null ? new BookDto() : bookDto);

    }

    /**
     * 获取书籍列表(微信编辑端)
     */
    @Override
    @RequestMapping(value = "getListPage4Wechat", method = RequestMethod.GET)
    public ResponseDto<PageBean> getListPage4Wechat( @CookieValue("userInfo") String userInfo,
                                                     @RequestParam(value = "currentPage") Integer currentPage,
                                                     @RequestParam(value = "numPerPage") Integer numPerPage,
                                                     @RequestParam(value = "bookName", required = false) String bookName,
                                                     @RequestParam(value = "channelId", required = false) Long channelId )
            throws BizException {
        Long adviserId = Cookie.getId(userInfo, Cookie._PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("bookName", bookName);
        paramMap.put("channelId", channelId);
        PageBean pageBean = bookBiz.getListPage4Wechat(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean == null ? new PageBean() : pageBean);
    }

    /**
     * 获取书籍列表(微信编辑端)
     */
    @Override
    @RequestMapping(value = "getlistPage", method = RequestMethod.GET)
    public ResponseDto<PageBean> listPage( @RequestHeader("token") String token,
                                           @RequestParam(value = "currentPage") Integer currentPage,
                                           @RequestParam(value = "numPerPage") Integer numPerPage,
                                           @RequestParam(value = "bookName", required = false) String bookName,
                                           @RequestParam(value = "channelId", required = false) Long channelId )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("bookName", bookName);
        paramMap.put("channelId", channelId);
        PageBean pageBean = bookBiz.getListPage4Wechat(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean == null ? new PageBean() : pageBean);
    }

    /**
     * 创建编辑书籍关联关系
     */
    @Override
    @RequestMapping(value = "setBookAdviserRelation4Wechat", method = RequestMethod.POST)
    public ResponseDto<?> setBookAdviserRelation4Wechat( @CookieValue("userInfo") String userInfo,
                                                         @RequestBody Book book ) throws BizException {
        Long adviserId = Cookie.getId(userInfo, Cookie._PARTY_ID);
        book.setCreatedUser(adviserId);
        bookBiz.setBookAdviserRelation(book);
        return new ResponseDto<>();
    }

    /**
     * 创建书籍
     */
    @Override
    @RequestMapping(value = "create4Wechat", method = RequestMethod.POST)
    public ResponseDto<BookDto> create4Wechat( @CookieValue("userInfo") String userInfo, @RequestBody Book book )
            throws BizException {
        Long adviserId = Cookie.getId(userInfo, Cookie._PARTY_ID);
        String systemCode = (String) Cookie.getUserInfo(userInfo).get(Cookie._SYSTEM_CODE);
        // 实体赋值
        book.setCreatedUser(adviserId);
        bookBiz.create(book, systemCode);
        BookDto bookDto = new BookDto();
        bookDto.setBookId(book.getBookId());
        return new ResponseDto<>(bookDto);
    }

    /**
     * 修改书籍(微信编辑端)
     */
    @Override
    @RequestMapping(value = "update4Wechat", method = RequestMethod.POST)
    public ResponseDto<BookDto> update4Wechat( @CookieValue("userInfo") String userInfo, @RequestBody Book book )
            throws BizException {
        Long adviserId = Cookie.getId(userInfo, Cookie._PARTY_ID);
        // 实体赋值
        book.setLastModifiedUser(adviserId);
        book.setCreatedUser(adviserId);
        bookBiz.updateByAdviser(book);
        return new ResponseDto<>();
    }

    /**
     * 根据ISBN获取期刊分页列表
     */
    @Override
    @RequestMapping(value = "getJournalPageByIsbn", method = RequestMethod.GET)
    public ResponseDto<?> getJournalPageByIsbn( @CookieValue("userInfo") String userInfo,
                                                @RequestParam(value = "typeCode", required = false) String typeCode,
                                                @RequestParam(value = "isbn", required = false) String isbn,
                                                @RequestParam(value = "currentPage") Integer currentPage,
                                                @RequestParam(value = "numPerPage") Integer numPerPage )
            throws BizException {
        if (currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        PageBean pageBean = bookBiz.getJournalPageByIsbn(new PageParam(currentPage, numPerPage), typeCode, isbn,
                channelId, wechatUserId);
        return new ResponseDto<>(pageBean);
    }

    /**
     * 获取书刊列表（微信客户端）
     */
    @Override
    @RequestMapping(value = "listPage4Wechat", method = RequestMethod.GET)
    public ResponseDto<PageBean> listPage4Wechat( @CookieValue("userInfo") String userInfo,
                                                  @RequestParam(value = "currentPage") Integer currentPage,
                                                  @RequestParam(value = "numPerPage") Integer numPerPage )
            throws BizException {
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        PageBean pageBean = bookBiz.listPage4Wechat(new PageParam(currentPage, numPerPage), channelId, wechatUserId);
        return new ResponseDto<>(pageBean);
    }

    /**
     * 获取编辑书刊列表-编辑首页
     */
    @Override
    @RequestMapping(value = "listBook", method = RequestMethod.GET)
    public ResponseDto<List<Object>> listBook( @RequestHeader("token") String token,
                                               @RequestParam(value = "number") Integer number )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        List<Object> books = bookBiz.listBook(adviserId, number);
        return new ResponseDto<>(books);
    }

    /**
     * 书刊列表-出版端
     */
    @Override
    @RequestMapping(value = "listPage4Agent", method = RequestMethod.GET)
    public ResponseDto<PageBean> listPage4Agent( @RequestHeader("token") String token,
                                                 @RequestParam(value = "bookName", required = false) String bookName,
                                                 @RequestParam(value = "channelId", required = false) Long channelId,
                                                 @RequestParam(value = "typeCode", required = false) String typeCode,
                                                 @RequestParam(value = "currentPage") Integer currentPage,
                                                 @RequestParam(value = "numPerPage") Integer numPerPage,
                                                 @RequestParam(value = "mainEditorName", required = false) String mainEditorName,
                                                 @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
                                                 @RequestParam(value = "isbn", required = false) String isbn,
                                                 @RequestParam(value = "templetId", required = false) Long templetId,
                                                 @RequestParam(value = "secondTempletId", required = false) Long secondTempletId,
                                                 @RequestParam(value = "isPrint", required = false) Boolean isPrint )
            throws BizException, PermissionException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("channelId", channelId);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("mainEditorName",
                mainEditorName != null && "".equals(mainEditorName.trim()) ? null : mainEditorName);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("isbn", isbn);
        paramMap.put("templetId", templetId);
        paramMap.put("secondTempletId", secondTempletId);
        paramMap.put("secondTempletIds", Optional.ofNullable(paramMap.get("secondTempletId")).map(Lists::newArrayList).orElse(Lists.newArrayList()));
        paramMap.put("isPrint", isPrint);
        return new ResponseDto<>(
                bookBiz.listPage4Agent(new PageParam(currentPage, numPerPage), paramMap, agentId));
    }

    /**
     * 获取编辑书刊列表-（APP编辑）
     */
    @Override
    @RequestMapping(value = "listBook4App", method = RequestMethod.GET)
    public ResponseDto<PageBean> listBook4App( @RequestHeader("token") String token,
                                               @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                               @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
                                               @RequestParam(value = "bookName", required = false) String bookName )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBean pageBean = bookBiz.listBook4App(adviserId, new PageParam(currentPage, numPerPage), bookName);
        return new ResponseDto<>(pageBean);
    }

    /**
     * 获取应用关联图书
     */
    @Override
    @RequestMapping(value = "listBookByAppId", method = RequestMethod.GET)
    @Deprecated
    public ResponseDto<List<BookDto>> listBookByAppId( @RequestHeader("token") String token,
                                                       @RequestParam(value = "appId", required = false) Long appId,
                                                       @RequestParam(value = "channelId", required = false) Long channelId )
            throws BizException {
        List<BookDto> bookDtos = bookBiz.listBookByAppId(appId, channelId);
        return new ResponseDto<>(bookDtos == null ? new ArrayList<>() : bookDtos);
    }

    /**
     * 获取书籍基础信息(包含统计信息)（出版端）
     */
    @Override
    @RequestMapping(value = "getById4Agent", method = RequestMethod.GET)
    public ResponseDto<BookDto> getById4Agent( @RequestHeader("token") String token,
                                               @RequestParam(value = "bookId") Long bookId,
                                               @RequestParam(value = "channelId", required = false) Long channelId,
                                               @RequestParam(value = "adviserId", required = false) Long adviserId )
            throws BizException {
        return new ResponseDto<>(bookBiz.getById(bookId, adviserId, channelId));
    }

    /**
     * 获取书籍基础信息（平台端）
     */
    @Override
    @RequestMapping(value = "getById4Platform", method = RequestMethod.GET)
    public ResponseDto<BookDto> getById4Platform( @RequestHeader("token") String token,
                                                  @RequestParam(value = "bookId") Long bookId,
                                                  @RequestParam(value = "channelId", required = false) Long channelId,
                                                  @RequestParam(value = "adviserId", required = false) Long adviserId )
            throws BizException {
        return new ResponseDto<>(bookBiz.getById4Platform(bookId, adviserId, channelId));
    }

    /**
     * 书刊回收站列表（30天内删除的书刊列表）
     */
    @Override
    @RequestMapping(value = "listPageDelete4Adviser", method = RequestMethod.GET)
    public ResponseDto<PageBean> listPageDelete4Adviser( @RequestHeader("token") String token,
                                                         @RequestParam(value = "currentPage") Integer currentPage,
                                                         @RequestParam(value = "numPerPage") Integer numPerPage )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        PageBean pageBean = bookBiz.listPageDelete4Adviser(adviserId, new PageParam(currentPage, numPerPage));
        return new ResponseDto<>(pageBean == null ? new PageBean() : pageBean);
    }

    /**
     * 获取图书关联统计个数(商品，应用，资源)
     */
    @Override
    @Deprecated
    @RequestMapping(value = "getCount4BookAssoc", method = RequestMethod.GET)
    public ResponseDto<BookAssocCount> getCount4BookAssoc( @RequestHeader("token") String token,
                                                           @RequestParam(value = "bookId") Long bookId,
                                                           @RequestParam(value = "channelId", required = false) Long channelId )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        BookAssocCount bookAssocCount = bookBiz.getCount4BookAssoc(bookId, channelId, adviserId);
        return new ResponseDto<>(bookAssocCount);
    }

    /**
     * 导出出版下所有书刊excel表（出版端）
     */
    @Override
    @RequestMapping(value = "exportBookToExcel", method = RequestMethod.GET)
    public ResponseDto<?> exportBookToExcel( @RequestHeader("token") String token )
            throws PermissionException, BizException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookBiz.exportBookToExcel(agentId);
        return new ResponseDto<>();
    }

    /**
     * 图书列表（平台端）
     */
    @Override
    @Permission(Code = ServerPermissionCode.book + "book/listBookClassify")
    @RequestMapping(value = "listBookClassify", method = RequestMethod.GET)
    public ResponseDto<PageBean> listBookClassify( @RequestHeader("token") String token,
                                                   @RequestParam(value = "typeCode", required = false) String typeCode,
                                                   @RequestParam(value = "templetId", required = false) Long templetId,
                                                   @RequestParam(value = "startDate", required = false) String startDate,
                                                   @RequestParam(value = "endDate", required = false) String endDate,
                                                   @RequestParam(value = "bookName", required = false) String bookName,
                                                   @RequestParam(value = "publish", required = false) String publish,
                                                   @RequestParam(value = "currentPage") Integer currentPage,
                                                   @RequestParam(value = "numPerPage") Integer numPerPage,
                                                   @RequestParam(value = "secondTempletId", required = false) Long secondTempletId,
                                                   @RequestParam(value = "isPrint", required = false) Boolean isPrint )
            throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("typeCode", typeCode);
        paramMap.put("templetId", templetId);
        paramMap.put("startDate", startDate);
        paramMap.put("endDate", endDate);
        paramMap.put("bookName", bookName);
        paramMap.put("publish", publish);
        paramMap.put("secondTempletId", secondTempletId);
        paramMap.put("secondTempletIds", Optional.ofNullable(paramMap.get("secondTempletId")).map(Lists::newArrayList).orElse(Lists.newArrayList()));
        if (!StringUtil.isEmpty(bookName)) {
            //去掉分词搜索20200331shuangli,by:zhuyajie
//            String[] bookNameList = bookName.trim().split("\\s+");
//            paramMap.put("bookNameList",Lists.newArrayList(bookNameList));
        }
        paramMap.put("isPrint", isPrint);
        PageBean pageBean = bookBiz.listBookClassify(new PageParam(currentPage, numPerPage), paramMap);
        return new ResponseDto<>(pageBean);
    }

    @GetMapping("exportBookClassify")
    @Override
    public ResponseDto<?> exportBookClassify(
            @RequestHeader("token") String token,
            @RequestParam(value = "typeCode", required = false) String typeCode,
            @RequestParam(value = "templetId", required = false) Long templetId,
            @RequestParam(value = "startDate", required = false) String startDate,
            @RequestParam(value = "endDate", required = false) String endDate,
            @RequestParam(value = "bookName", required = false) String bookName,
            @RequestParam(value = "publish", required = false) String publish,
            @RequestParam(value = "secondTempletId", required = false) Long secondTempletId
    ) throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("typeCode", typeCode);
        paramMap.put("templetId", templetId);
        paramMap.put("startDate", startDate);
        paramMap.put("endDate", endDate);
        paramMap.put("bookName", bookName);
        paramMap.put("publish", publish);
        paramMap.put("secondTempletId", secondTempletId);
        paramMap.put("secondTempletIds", Optional.ofNullable(paramMap.get("secondTempletId")).map(Lists::newArrayList).orElse(Lists.newArrayList()));
        if (!StringUtil.isEmpty(bookName)) {
            String[] bookNameList = bookName.trim().split("\\s+");
            paramMap.put("bookNameList", Lists.newArrayList(bookNameList));
        }
        bookBiz.exportBookClassify(paramMap);
        return new ResponseDto<>();
    }

    /**
     * 导出平台下所有书刊excel表（平台端）
     */
    @Override
    @RequestMapping(value = "exportBookToExcel4Platform", method = RequestMethod.GET)
    public ResponseDto<?> exportBookToExcel4Platform( @RequestHeader("token") String token )
            throws PermissionException, BizException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookBiz.exportBookToExcel4Platform();
        return new ResponseDto<>();
    }

    /**
     * 判断书籍ISBN码是否存在,存在则获取书籍信息
     */
    @Override
    @RequestMapping(value = "getByIsbnAndTypeCode", method = RequestMethod.GET)
    public ResponseDto<BookDto> getByIsbnAndTypeCode( @RequestHeader("token") String token,
                                                      @RequestParam(value = "isbn") String isbn,
                                                      @RequestParam(value = "typeCode") String typeCode )
            throws BizException {
        BookDto bookDto = bookBiz.getByIsbnAndTypeCode(isbn, typeCode);
        return new ResponseDto<>(bookDto == null ? new BookDto() : bookDto);
    }

    /**
     * 编辑新增书籍
     */
    @Override
    @RequestMapping(value = "create4Adviser", method = RequestMethod.POST)
    public ResponseDto<BookDto> create4Adviser( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.TENANT_ID);
        // 实体赋值
        book.setCreatedUser(adviserId);
        book.setAgentId(agentId);
        BookDto bookDto = bookBiz.create4Adviser(book);
        return new ResponseDto<>(bookDto);
    }

    @Override
    @RequestMapping(value = "getByIsbnAndSerialNumber", method = RequestMethod.GET)
    public ResponseDto<BookDto> getByIsbnAndSerialNumber( @RequestParam(value = "isbn", required = false) String isbn,
                                                          @RequestParam(value = "serialNumber", required = false) String serialNumber ) throws BizException {
        BookDto book = bookBiz.getByIsbn(isbn, serialNumber);
        return new ResponseDto<>(book == null ? new BookDto() : book);
    }

    /**
     * 修改图书类型与图书基本信息（编辑）
     */
    @Override
    @RequestMapping(value = "updateBookAndBookType", method = RequestMethod.POST)
    public ResponseDto<?> updateBookAndBookType( @RequestHeader("token") String token, @RequestBody Book book )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        // 实体赋值
        book.setLastModifiedUser(adviserId);
        bookBiz.updateBookAndBookType(book);
        return new ResponseDto<>();
    }

    @Override
    @RequestMapping(value = "checkISBN4App", method = RequestMethod.GET)
    public ResponseDto<Boolean> checkISBN4App( @RequestHeader("token") String token,
                                               @RequestParam(value = "isbn") String isbn ) throws PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Boolean bool = bookBiz.checkISBN4App(isbn);
        return new ResponseDto<>(bool);
    }

    /**
     * 根据渠道ID分页获取图书列表
     */
    @Override
    @GetMapping("getBookListByChannelId4Adviser")
    public ResponseDto<PageBean> getBookListByChannelId4Adviser( @RequestHeader("token") String token,
                                                                 @RequestParam(value = "currentPage") Integer currentPage,
                                                                 @RequestParam(value = "numPerPage") Integer numPerPage,
                                                                 @RequestParam(value = "channelId") Long channelId,
                                                                 @RequestParam(value = "name", required = false) String name ) throws PermissionException, BizException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        return new ResponseDto<>(bookBiz.getBookBaseInfoListByChannelId4Adviser(partyId, channelId, pageParam, name));
    }

    @RequestMapping(value = "listPage4PC", method = RequestMethod.GET)
    public ResponseDto<PageBean> getListPage4PC( @RequestHeader("token") String token,
                                                 @RequestParam(value = "name", required = false) String name,
                                                 @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                 @RequestParam(value = "numPerPage", required = false) Integer numPerPage )
            throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        PageBean pageBean = bookBiz.getListPage4PC(new PageParam(currentPage, numPerPage), name);
        return new ResponseDto<PageBean>(null == pageBean ? new PageBean() : pageBean);
    }

    @Override
    @RequestMapping(value = "listNoAuthBook", method = RequestMethod.POST)
    public ResponseDto<PageBeanNew> listNoAuthBook( @RequestHeader("token") String token,
                                                    @RequestBody ListNoAuthBookParam listNoAuthBookParam )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew pageBeanNew = bookBiz.listNoAuthBook(listNoAuthBookParam, adviserId);
        return new ResponseDto<PageBeanNew>(null == pageBeanNew ? new PageBeanNew() : pageBeanNew);
    }

    @Override
    @RequestMapping(value = "listNoAuthGroupBook", method = RequestMethod.POST)
    public ResponseDto<PageBeanNew> listNoAuthGroupBook( @RequestHeader("token") String token,
                                                         @RequestBody ListNoAuthGroupBookParam listNoAuthGroupBookParam )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew pageBeanNew = bookBiz.listNoAuthGroupBook(listNoAuthGroupBookParam, adviserId);
        return new ResponseDto<PageBeanNew>(null == pageBeanNew ? new PageBeanNew() : pageBeanNew);
    }

    @Override
    @RequestMapping(value = "getListPage4SelfBookGroup", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<AdviserBookInfoDTO>> getListPage4SelfBookGroup(
            @RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "graLabelId", required = false) Long graLabelId,
            @RequestParam(value = "subLabelId", required = false) Long subLabelId,
            @RequestParam(value = "verLabelId", required = false) Long verLabelId,
            @RequestParam(value = "areaLabelId", required = false) Long areaLabelId,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage )
            throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        PageBeanNew<AdviserBookInfoDTO> pageBean = bookBiz.getListPage4SelfBookGroup(currentPage, numPerPage, name, null, graLabelId, subLabelId, verLabelId, areaLabelId);
        return new ResponseDto<>(null == pageBean ? new PageBeanNew<>() : pageBean);
    }

    @Override
    @RequestMapping(value = "getListPage4SelfBookGroupByAdvise", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<AdviserBookInfoDTO>> getListPage4SelfBookGroupByAdvise(
            @RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        PageBeanNew<AdviserBookInfoDTO> pageBean = bookBiz.getListPage4SelfBookGroup(currentPage, numPerPage, name, adviserId, null, null, null, null);
        return new ResponseDto<>(null == pageBean ? new PageBeanNew<>() : pageBean);
    }

    @GetMapping("getAdviserBooks4Pcloud")
    @Override
    public ResponseDto<?> getAdviserBooks4Pcloud(
            @RequestHeader("token") String token,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "agentId", required = false) Long agentId,
            @RequestParam(value = "templetId", required = false) Long templetId,
            @RequestParam(value = "secondTempletId", required = false) Long secondTempletId,
            @RequestParam(value = "typeCode", required = false) String typeCode,
            @RequestParam(value = "currentPage") Integer currentPage,
            @RequestParam(value = "numPerPage") Integer numPerPage
    ) throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        return new ResponseDto<>(bookBiz.getAdviserBooks4Pcloud(keyword, agentId, templetId, secondTempletId, typeCode, currentPage, numPerPage));
    }

    @GetMapping("getAdviserBooks4H5")
    @Override
    public ResponseDto<?> getAdviserBooks4H5(
            @RequestHeader(value = "Gray-Status", required = false) String grayStatus,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "templetId", required = false) Long templetId,
            @RequestParam(value = "secondTempletId", required = false) Long secondTempletId,
            @RequestParam(value = "currentPage") Integer currentPage,
            @RequestParam(value = "numPerPage") Integer numPerPage
    ) throws BizException, PermissionException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        return new ResponseDto<>(bookBiz.getAdviserBooks4H5(grayStatus, keyword, templetId, secondTempletId, currentPage, numPerPage));
    }

    @GetMapping("getBookLabels")
    @Override
    public ResponseDto<?> getBookLabels(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookBiz.getBookLabels(partyId));
    }

    /**
     * 书刊收益列表-出版端
     */
    @Override
    @RequestMapping(value = "listPageIncomeDetail", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew> listPageIncomeDetail( @RequestHeader("token") String token,
                                                          @RequestParam(value = "keyWord", required = false) String keyWord,
                                                          @RequestParam(value = "channelId", required = false) Long channelId,
                                                          @RequestParam(value = "typeCode", required = false) String typeCode,
                                                          @RequestParam(value = "currentPage") Integer currentPage,
                                                          @RequestParam(value = "numPerPage") Integer numPerPage,
                                                          @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
                                                          @RequestParam(value = "templetId", required = false) Long templetId,
                                                          @RequestParam(value = "secondTempletId", required = false) Long secondTempletId )
            throws BizException, PermissionException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("keyWord", keyWord != null && "".equals(keyWord.trim()) ? null : keyWord);
        paramMap.put("channelId", channelId);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("templetId", templetId);
        paramMap.put("secondTempletId", secondTempletId);
        paramMap.put("secondTempletIds", Optional.ofNullable(paramMap.get("secondTempletId")).map(Lists::newArrayList).orElse(Lists.newArrayList()));
        return new ResponseDto<>(bookBiz.listPageIncomeDetail(new PageParam(currentPage, numPerPage), paramMap, agentId));
    }

    @ApiOperation("导出书刊收益")
    @GetMapping("exportBookIncomeDetail")
    public ResponseDto<?> exportBookIncomeDetail(
            @RequestHeader("token") String token,
            @RequestParam(value = "keyWord", required = false) String keyWord,
            @RequestParam(value = "channelId", required = false) Long channelId,
            @RequestParam(value = "typeCode", required = false) String typeCode,
            @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
            @RequestParam(value = "templetId", required = false) Long templetId,
            @RequestParam(value = "secondTempletId", required = false) Long secondTempletId ) throws PermissionException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.agent.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        String isSystem = (String) map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)) {
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("keyWord", keyWord != null && "".equals(keyWord.trim()) ? null : keyWord);
        paramMap.put("channelId", channelId);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("templetId", templetId);
        paramMap.put("secondTempletId", secondTempletId);
        paramMap.put("secondTempletIds", Optional.ofNullable(paramMap.get("secondTempletId")).map(Lists::newArrayList).orElse(Lists.newArrayList()));
        bookBiz.exportBookIncomeDetail(paramMap, agentId, partyId, systemCode);
        return new ResponseDto<>();
    }

    @Override
    @GetMapping("getBookAndServeList4Channel")
    public ResponseDto<?> getBookAndServeList4Channel(
            @RequestHeader("token") String token,
            @RequestParam(value = "currentPage") Integer currentPage,
            @RequestParam(value = "numPerPage") Integer numPerPage,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
            @RequestParam(value = "startDate", required = false) String startDate,
            @RequestParam(value = "endDate", required = false) String endDate
    ) throws PermissionException, JsonParseException, BizException {
        Long channelId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew<BookDto> pageBean = bookBiz.getBookAndServeList4Channel(currentPage, numPerPage, channelId, name, isFundSupport, startDate, endDate);
        return new ResponseDto<>(pageBean);
    }

    @ApiOperation("获取书刊")
    @GetMapping("getAdviserBook")
    @Override
    public ResponseDto<?> getAdviserBook(
            @RequestParam(value = "bookId") Long bookId,
            @RequestParam(value = "adviserId") Long adviserId,
            @RequestParam(value = "channelId") Long channelId
    ) throws BizException, PermissionException {
        return new ResponseDto<>(bookBiz.getAdviserBook(bookId, adviserId, channelId));
    }


    @ApiOperation("添加移除质检书刊")
    @Override
    @RequestMapping(value = "updateBookQualifyInfo", method = RequestMethod.POST)
    public ResponseDto<?> updateBookQualifyInfo( @RequestHeader("token") String token, @RequestBody BookQualifyInfoVO bookQualifyInfo ) throws PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookBiz.updateBookQualifyInfo(bookQualifyInfo, adviserId);
        return new ResponseDto<>();

    }


    //添加书刊时书籍查询 入参bookName isbn   查询更新后质检书刊 isShowQualifyInfo
    @Override
    @GetMapping("getAddBookQualifyList")
    public ResponseDto<?> getAddBookQualifyList(
            @RequestHeader("token") String token,
            @RequestParam(value = "currentPage") Integer currentPage,
            @RequestParam(value = "numPerPage") Integer numPerPage,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "isShowQualifyInfo", required = false) Integer isShowQualifyInfo
    ) throws PermissionException, JsonParseException, BizException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew<BookQualifyVO> pageBean = bookBiz.getAddBookQualifyList(currentPage, numPerPage, name, isShowQualifyInfo, adviserId);
        return new ResponseDto<>(pageBean);
    }


    @Override
    @GetMapping("getBookInfoByBookGroupId")
    public ResponseDto<?> getBookInfoByBookGroupId(
            @RequestParam(value = "bookGroupId", required = true) Long bookGroupId
    ) throws PermissionException, JsonParseException, BizException {
        SearchBookVO bookVO = bookBiz.getBookInfoByBookGroupId(bookGroupId);
        return new ResponseDto<>(bookVO);
    }

    @Override
    @RequestMapping(value = "updateIsOpenRobotProcess", method = RequestMethod.POST)
    public ResponseDto<?> updateIsOpenRobotProcess(@RequestHeader("token") String  token, @RequestBody UpdateBookRobotProcessDto robotProcessDto) throws PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token,SessionUtil.PARTY_ID);
        bookBiz.updateIsOpenRobotProcess(adviserId,robotProcessDto);
        return new ResponseDto<>();
    }

    @Override
    @RequestMapping(value = "updateIsOpenRobotProcess4Now", method = RequestMethod.POST)
    public ResponseDto<?> updateIsOpenRobotProcess4Now(@RequestHeader("token") String  token, @RequestBody UpdateBookRobotProcessDto robotProcessDto) throws PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token,SessionUtil.PARTY_ID);
        bookBiz.updateIsOpenRobotProcess4Now(adviserId,robotProcessDto);
        return new ResponseDto<>();
    }

    @GetMapping("getAdviserBooks4Rights")
    public ResponseDto<?> getAdviserBooks4Rights(
            @RequestHeader("token") String token,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam(value = "agentId", required = false) Long agentId,
            @RequestParam(value = "templetId", required = false) Long templetId,
            @RequestParam(value = "secondTempletId", required = false) Long secondTempletId,
            @RequestParam(value = "typeCode", required = false) String typeCode,
            @RequestParam(value = "currentPage") Integer currentPage,
            @RequestParam(value = "numPerPage") Integer numPerPage
    ) throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        return new ResponseDto<>(bookBiz.getAdviserBooks4Rights(keyword, agentId, templetId, secondTempletId, typeCode, currentPage, numPerPage));
    }

    @Override
    @PostMapping("getAdviserBooks4Applet")
    public ResponseDto<?> getAdviserBooks4Applet(
            @RequestBody @Validated BookSearchParamVO bookSearchParamVO
    ) throws BizException, PermissionException {
        return new ResponseDto<>(bookBiz.getAdviserBooks4Applet(bookSearchParamVO));
    }

    @GetMapping("getBookLabels4Erp")
    public ResponseDto<?> getBookLabels4Erp(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException {
        return new ResponseDto<>(bookBiz.getBookLabels4Erp());
    }
}