package com.pcloud.common.utils.cache.redis;

import com.pcloud.common.utils.json.JSONUtils;
import com.pcloud.common.utils.string.StringUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisCluster;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * 
 * @描述：redis集群初始化
 * @作者：songx
 * @创建时间：2017年9月13日,下午5:50:17
 * @版本：1.0
 */
public class JedisClusterBeanBuilder {

	/**
	 * 
	 */
	private static final Logger LOGGER = LoggerFactory.getLogger(JedisClusterBeanBuilder.class);


	/**
	 * 正则校验IP
	 */
	private static final Pattern p = Pattern.compile("^.+[:]\\d{1,5}\\s*$");
	
	/**
	 * Jedis连接池
	 * 
	 * @return
	 */
	public static GenericObjectPoolConfig<?> createJedisPoolConfig(JedisClusterProperties properties) {
		LOGGER.info("jedis pool config initialize...");

		GenericObjectPoolConfig<?> poolConfig = new GenericObjectPoolConfig<>();
		poolConfig.setMaxTotal(properties.getPool().getMaxTotal());
		poolConfig.setMinIdle(properties.getPool().getMinIdle());
		poolConfig.setMaxIdle(properties.getPool().getMaxIdle());
		poolConfig.setMaxWaitMillis(properties.getPool().getMaxWaitMillis());
		poolConfig.setTimeBetweenEvictionRunsMillis(properties.getPool().getTimeBetweenEvictionRunsMillis());
		poolConfig.setTestOnBorrow(properties.getPool().isTestOnBorrow());

		LOGGER.info("jedis pool config initialize,properties={}", JSONUtils.toJsonString(properties));
		return poolConfig;
	}
	
	/**
	 * JedisCluster
	 */
	public static JedisCluster createJedisCluster(JedisClusterProperties properties, GenericObjectPoolConfig<?> config) {
		LOGGER.info("JedisCluster create start,{},=>{}<=]", properties.getTimeout(), properties.getPassword());
		try {
			JedisCluster jc = new JedisCluster(parseHostAndPort(properties.getHost()),
					properties.getTimeout(), properties.getTimeout(), properties.getMaxRedirections(), StringUtils.trimToNull(properties.getPassword()), config);

			LOGGER.info("JedisCluster create completed,echo={}", jc.echo("Echo Successfully."));

			return jc;
		} catch (Exception e) {
			LOGGER.info("JedisCluster create without password,{},=>{}<=],message={}", properties.getTimeout(), properties.getPassword(), e.getMessage());

			JedisCluster jc = new JedisCluster(parseHostAndPort(properties.getHost()), properties.getTimeout(), properties.getMaxRedirections(), config);

			LOGGER.info("JedisCluster create without password,echo={}", jc.echo("Echo Successfully."));
			return jc;
		}

	}

	/**
	 * 组装节点IP
	 * 
	 * @return
	 */
	private static Set<HostAndPort> parseHostAndPort(final String host) {
		LOGGER.info("JedisCluster connect nodes {}", host);
		Set<HostAndPort> hostAndPorts = new HashSet<>();
		if (StringUtil.isEmpty(host)) {
			throw new IllegalArgumentException("解析 jedis配置文件 -> 节点IP不能为空");
		}
		String[] hosts = host.split(",");
		for (int i = 0; i < hosts.length; i++) {
			String value = hosts[i];
			boolean isIpPort = p.matcher(value).matches();
			if (!isIpPort) {
				throw new IllegalArgumentException("解析 jedis配置文件 -> ip或 port不合法");
			}
			String[] ipAndPort = value.split(":");
			hostAndPorts.add(new HostAndPort(ipAndPort[0].trim(), Integer.parseInt(ipAndPort[1].trim())));
		}
		return hostAndPorts;
	}

}
