package com.pcloud.common.utils.robot;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 企业微信Web Hook 机器人
 */
public class WeWorkWebHookRobotUtils {

    private WeWorkWebHookRobotUtils() {
    }

    private static final String REQUEST_URL = "https://qyapi.weixin.qq.com/cgi-bin/webhook/send?key=";

    /**
     * 推送文本类型消息
     *
     * @param robotKey    机器人key
     * @param content     文本消息内容
     * @param userIds     userid的列表，提醒群中的指定成员(@某个成员)，@all表示提醒所有人
     * @param userMobiles 手机号列表，提醒手机号对应的群成员(@某个成员)，@all表示提醒所有人
     * @return 推送结果
     */
    public static boolean sendTextMsg(String robotKey, String content, List<String> userIds, List<String> userMobiles) {
        if (StrUtil.isBlank(robotKey) || StrUtil.isBlank(content)) {
            return false;
        }

        Map<String, Object> params = new HashMap<>();
        params.put("msgtype", "text");

        Map<String, Object> text = new HashMap<>();
        text.put("content", content);
        if (CollUtil.isNotEmpty(userIds)) {
            text.put("mentioned_list", userIds);
        }
        if (CollUtil.isNotEmpty(userMobiles)) {
            text.put("mentioned_mobile_list", userIds);
        }
        params.put("text", text);

        return postMsg(robotKey, params);
    }


    /**
     * 发送 markdown 消息
     *
     * @param robotKey 机器人Key
     * @param content  markdown内容
     * @return 推送结果
     */
    public static boolean sendMarkdownMsg(String robotKey, String content) {
        Map<String, Object> params = new HashMap<>();
        params.put("msgtype", "markdown");

        Map<String, Object> markdown = new HashMap<>();
        markdown.put("content", content);
        params.put("markdown", markdown);

        return postMsg(robotKey, params);
    }

    /**
     * 发送 图片消息
     *
     * @param robotKey 机器人key
     * @param base64   图片的base64编码
     * @param md5      图片base64编码前的md5值
     * @return 推送结果
     */
    public static boolean sendImageMsg(String robotKey, String base64, String md5) {
        Map<String, Object> params = new HashMap<>();
        params.put("msgtype", "image");

        Map<String, Object> image = new HashMap<>();
        image.put("base64", base64);
        image.put("md5", md5);
        params.put("image", image);

        return postMsg(robotKey, params);
    }

    /**
     * 发送图文消息
     *
     * @param robotKey 机器人key
     * @param articles 图文内容，一个图文消息支持1到8条图文
     * @return 推送结果
     */
    public static boolean sendNewsMsg(String robotKey, List<Article> articles) {
        Map<String, Object> params = new HashMap<>();
        params.put("msgtype", "news");

        Map<String, Object> news = new HashMap<>();
        news.put("articles", articles);
        params.put("news", news);

        return postMsg(robotKey, params);
    }

    private static boolean postMsg(String robotKey, Map<String, Object> params) {
        try {
            String result = HttpUtil.post(REQUEST_URL + robotKey, JSONObject.toJSONString(params));
            if (StrUtil.isNotBlank(result) && Objects.equals(JSONObject.parseObject(result).getInteger("errcode"), 0)) {
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return false;
    }


    public static class Article {
        private String title;
        private String description;
        private String url;
        private String picurl;

        public Article(String title, String description, String url, String picurl) {
            this.title = title;
            this.description = description;
            this.url = url;
            this.picurl = picurl;
        }

        public Article() {
        }

        public String getTitle() {
            return title;
        }

        public void setTitle(String title) {
            this.title = title;
        }

        public String getDescription() {
            return description;
        }

        public void setDescription(String description) {
            this.description = description;
        }

        public String getUrl() {
            return url;
        }

        public void setUrl(String url) {
            this.url = url;
        }

        public String getPicurl() {
            return picurl;
        }

        public void setPicurl(String picurl) {
            this.picurl = picurl;
        }
    }

}
