package com.pcloud.common.core.biz;

import java.io.IOException;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Session;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.jms.core.MessageCreator;
import org.springframework.stereotype.Component;

import com.pcloud.common.core.dto.SendEmailDto;
import com.pcloud.common.core.dto.SendMessageDto;
import com.pcloud.common.core.dto.SendNotifyDto;
import com.pcloud.common.utils.mq.MqMessage;
import com.pcloud.common.utils.mq.MqMessageTypeEnum;

/**
 * 
 * @描述：邮件业务实现
 * @作者：shichunshan
 * @创建时间：2016年6月24日,下午2:19:27
 * @版本：1.0
 */
@Component("mailBiz")
public class MessageBizImpl implements MessageBiz{
	
	private final static Logger logger=LoggerFactory.getLogger(MessageBizImpl.class);

	/**
	 * 消息模板
	 */
	@Autowired(required = false)
	@Qualifier("jmsMessageTemplate")
	private JmsTemplate jmsMessageTemplate;

	@Override
	public void sendEmail(SendEmailDto sendEmailDto) {
		MessageCreator messageCreator = new MessageCreator() {

			@Override
			public Message createMessage(Session session) throws JMSException {
				MqMessage mqMessage = null;
				try {
					mqMessage = formateMessage(sendEmailDto);
					logger.info("mqMessage+++++++++++" + mqMessage);
				} catch (IOException e) {
					logger.error("邮件发送失败，" + sendEmailDto);
					return null;
				}
				return session.createObjectMessage(mqMessage);
			}
		};
		try {
			jmsMessageTemplate.send(messageCreator);
		} catch (Exception e) {
			logger.error("邮件发送失败，" + e.getMessage() + "," + sendEmailDto, e);
			return;
		}

		logger.info("邮件发送成功，" + sendEmailDto);
	}

	/**
	 * 格式化邮件内容
	 * 
	 * @param mailParam
	 *            邮件参数
	 * @return 队列消息
	 * @throws IOException
	 *             序列化为json错误
	 */
	private MqMessage formateMessage(SendEmailDto sendEmailDto) throws IOException {
		ObjectMapper objectMapper = new ObjectMapper();
		String content = objectMapper.writeValueAsString(sendEmailDto);
		return new MqMessage(MqMessageTypeEnum.EMAIL, content);
	}

	@Override
	public void sendLetter(SendNotifyDto sendNotifyDto) {
		MessageCreator messageCreator = new MessageCreator() {

			@Override
			public Message createMessage(Session session) throws JMSException {
				MqMessage mqMessage = null;
				try {
					mqMessage = formateMessage(sendNotifyDto);
					logger.info("mqMessage+++++++++++" + mqMessage);
				} catch (IOException e) {
					logger.error("站内信发送失败，" + sendNotifyDto);
					return null;
				}
				return session.createObjectMessage(mqMessage);
			}
		};
		try {
			jmsMessageTemplate.send(messageCreator);
		} catch (Exception e) {
			logger.error("站内信发送失败，" + e.getMessage() + "," + sendNotifyDto, e);
			return;
		}

		logger.info("站内信发送成功，" + sendNotifyDto);
	}

	/**
	 * 格式化站内信内容
	 * 
	 * @param sendNotifyDto
	 *            站内信参数
	 * @return 队列消息
	 * @throws IOException
	 *             序列化为json错误
	 */
	private MqMessage formateMessage(SendNotifyDto sendNotifyDto) throws IOException {
		ObjectMapper objectMapper = new ObjectMapper();
		String content = objectMapper.writeValueAsString(sendNotifyDto);
		return new MqMessage(MqMessageTypeEnum.LETTER, content);
	}

	@Override
	public void sendSMS(SendMessageDto smsParam) {
		MessageCreator messageCreator = new MessageCreator() {

			@Override
			public Message createMessage(Session session) throws JMSException {
				MqMessage mqMessage = null;
				try {
					mqMessage = formateMessage(smsParam);
					logger.info("mqMessage+++++++++++" + mqMessage);
				} catch (IOException e) {
					logger.error("短信发送失败，" + smsParam);
					return null;
				}
				return session.createObjectMessage(mqMessage);
			}
		};
		try {
			jmsMessageTemplate.send(messageCreator);
		} catch (Exception e) {
			logger.error("短信发送失败，" + e.getMessage() + "," + smsParam, e);
			return;
		}

		logger.info("短信发送成功，" + smsParam);
	}

	/**
	 * 格式化内容
	 * 
	 * @param sendSMSDto
	 *            邮件参数
	 * @return 队列消息
	 * @throws IOException
	 *             序列化为json错误
	 */
	private MqMessage formateMessage(SendMessageDto smsParam) throws IOException {
		ObjectMapper objectMapper = new ObjectMapper();
		String content = objectMapper.writeValueAsString(smsParam);
		return new MqMessage(MqMessageTypeEnum.SHORT_MESSAGE, content);
	}
	
}
